

*Optional: set the directory to the location of this do file
cd

*This program reads in raw data and saves a dta file for each version of the cleaned data


capture program drop MakeData
program define MakeData

	******************
	* Parameters
	******************

	*set to 1 to do an adjustment for Chinese New Years. Set to 0 to not do the adjustment
	local ChineseNewYearAdj = `1'

	*set to 1 to use the biweight filter. Set to 0 to not use it.
	local UseBW = `2'
	*specify the biweight filter parameter. We try 36 and 120. Set to "" if not using the BW filter
	local BWParm = "`3'"

	*set to start and end dates for the transformed data. This is also the start and end dates for the biweight filter
	local StartY `4'
	local StartM `5'
	local EndY   `6'
	local EndM   `7'




	* Other locals
	******************
	*set to folder where we can find the STATA interface for the x12-ARIMA program
	adopath ++ "../programs/X12_ARIMA"

	*set to "1" to use month-over-month growth rates. Set to "12" to use year-over-year (that is, 12-month) growth rates
	local GrowthType = "1"


	if `GrowthType'==1{
		*set to 1 to seasonally adjust the data in levels. IF using MOM growth rates, you probably want to seasonally adjust the data.
		local SeasonallyAdjust = 1
	}
	if `GrowthType'==12{
		local SeasonallyAdjust = 0
	}

	if `GrowthType'==1{
		local MOMorYOY MOM
	}
	if `GrowthType'==12{
		local MOMorYOY YOY
	}

	if `ChineseNewYearAdj'==0{
		local CNYadj noCNYadj
	}
	if `ChineseNewYearAdj'==1{
		local CNYadj CNYadj
	}

	if `UseBW'==0{
		local BW = ""
	}
	if `UseBW'==1{
		local BW = " BW"
	}




	**********************************
	* Read in data for the overnight rate (from FRED)
	**********************************
	local FREDVars Overnight
	foreach v in `FREDVars'{
		import excel using "../raw data/raw_data.xlsx", clear sheet("`v'")
		drop if _n<=19
		rename A date_str
		rename B Overnight
		
		*generate STATA date variable
		gen date = mofd(date(date_str, "DMY"))
		format date %tm	
		
		drop date_str
		save "../clean data/temp_files/`v'.dta", replace
	}





	******************************
	* Read in variables that only go into broadest (41 variables total, but we will drop some of these)
	*******************************
	*note that the following sheets contain monthly data:
	local MonthlySheets Sheet1 Sheet2 Sheet3 Sheet4 Sheet5 Sheet6 Sheet7 Sheet8 Sheet19 Sheet22 Sheet23 Sheet24 Sheet25 Sheet26 Sheet27 Sheet28 Sheet29 Sheet30 Sheet31 Sheet32 Sheet36 Sheet37 Sheet38 Sheet39 Sheet40 Sheet41 Sheet42 Sheet43 Sheet44 Sheet45 Sheet46 Sheet47 Sheet48 Sheet49 Sheet50 Sheet51 Sheet52 Sheet53 Sheet54 Sheet55 Sheet56

	foreach s in `MonthlySheets'{
		import excel using "../raw data/raw_data.xlsx", firstrow clear sheet("`s'")
		rename SelectthislinkandclickRefre date_str
		
		*drop lines from the header. drop empty lines at the end
		drop if _n<=10
		drop if date_str==""
		
		*generate a STATA date variable
		gen date = mofd(date(date_str, "DMY"))
		format date %tm
		
		***Special adjustments to data***
		if "`s'"=="Sheet3"{
			rename CNGovernmentSurplusorDefici GovtSurp
		}
		if "`s'"=="Sheet6"{
			rename CNGovernmentSurplusorDefici GovtSurpCentral
		}
		if "`s'"=="Sheet40"{
			rename CNNominalLendingRateShort CNNominalRate6mo
		}
		if "`s'"=="Sheet41"{
			rename CNNominalLendingRateShort CNNominalRate1yr
		}
		
		drop date_str
		save "../clean data/temp_files/`s'.dta", replace
	}

	*merge all sheets together
	local FirstSheet = 1
	foreach s in `MonthlySheets'{
		
		if `FirstSheet'==1{
			use "../clean data/temp_files/`s'.dta", clear
		}
		
		if `FirstSheet'==0{
			merge 1:1 date using "../clean data/temp_files/`s'.dta"
			drop _merge
		}
		
		local FirstSheet = 0
	}

	save "../clean data/temp_files/broadest_only.dta", replace






	*****************************
	* Read in remaining China variables
	*****************************
	local BroadVars GasConsumption ConsumerIndex Retail IP ElectricityConsumption ElectricityProduction Rail Property_RealEstate CrudeSteel Trucks PMI PMI_Export FloorSpace Lending RawMaterials BenchmarkRate20d
	foreach v in `BroadVars'{
		disp "This variable is: `v'"
		
		import excel using "../raw data/raw_data.xlsx", clear sheet(`v')
		*sometimes an extra columns C or D appears that is all missing. Drop it.
		keep A B
		
		rename A date_str
		rename B `v'
		
		if "`v'"!="GasConsumption"{
			*drop lines from the header. drop empty lines at the end
			drop if _n<=11
			drop if date_str==""
			
			*generate a STATA date variable
			gen date = mofd(date(date_str, "DMY"))
			format date %tm
		}
		
		if "`v'"=="GasConsumption"{
			*drop lines from the header. drop empty lines at the end
			drop if _n<=2
			drop if date_str==""
			
			*generate a STATA date variable
			gen year  = substr(date_str,-4,4)
			gen month = substr(date_str,1,2)
			replace month = subinstr(month,"/","",.)
			destring year, replace
			destring month, replace
			gen date = ym(year,month)
			format date %tm
			drop year month
		}
		
		if "`v'"=="RawMaterials"{
			*linearly interpolate to monthly data
			tsset date
			tsfill
			
			destring `v', replace
			ipolate RawMaterials date, generate(`v'_monthly)
			
			drop `v'
			rename `v'_monthly `v'
		}
		
		drop date_str
		save "../clean data/temp_files/`v'.dta", replace
	}

	*merge all sheets together
	local FirstSheet = 1
	foreach v in `BroadVars'{
		
		if `FirstSheet'==1{
			use "../clean data/temp_files/`v'.dta", clear
		}
		if `FirstSheet'==0{
			merge 1:1 date using "../clean data/temp_files/`v'.dta"
			drop _merge
		}
		
		local FirstSheet = 0
	}

	save "../clean data/temp_files/broad.dta", replace




	***************************
	* read in data on foreign shocks
	****************************
	*these data are from Haver
	local ForeignShocks IP_US Commodity Oil
	foreach v in `ForeignShocks'{
		disp "This variable is: `v'"
		
		import excel using "../raw data/raw_data.xlsx", clear sheet(`v')
		drop if _n==1
		rename A date_str
		rename B `v'
		
		*generate a STATA date variable
		gen year  = substr(date_str,1,4)
		gen month = substr(date_str,-2,2)
		destring year, replace
		destring month, replace
		gen date = ym(year,month)
		format date %tm
		drop year month date_str
		
		save "../clean data/temp_files/`v'.dta", replace
	}

	*merge foreign shocks together
	local FirstSheet = 1
	foreach v in `ForeignShocks'{

		if `FirstSheet'==1{
			use "../clean data/temp_files/`v'.dta", clear
		}
		if `FirstSheet'==0{
			merge 1:1 date using "../clean data/temp_files/`v'.dta"
			drop _merge
		}
		
		local FirstSheet = 0
	}
	save "../clean data/temp_files/ForeignShocks.dta", replace




	***********************
	* merge all variables together & clean
	************************
	use "../clean data/temp_files/broad.dta", clear
	merge 1:1 date using "../clean data/temp_files/broadest_only.dta"
		*obs that didn't merge are pre1990 or late 2013
	drop _merge

	merge 1:1 date using "../clean data/temp_files/Overnight.dta"
		*only pre-1990 and late 2013 obs are not matching, as expected
	drop _merge

	merge 1:1 date using "../clean data/temp_files/ForeignShocks.dta"
		*only late 2013 obs didn't match, as expected
	drop _merge



	*recreate more readable mnenomics
	rename CN* *
	rename GovtRevenueCentralLevel    GovtRevenueCentral
	rename GovtExpenditureCentralLe   GovtExpenditureCentral
	rename HoldingsofUSTreasurySecu   HoldingsofUSTreasury
	rename NoofEmployeeIndustrialE    EmpIndustrial
	rename ConsumerPriceIndexMoM      CPI_MOM
	rename CPIMoMCoreexclFood         CPICore_MOM
	rename CPIMoMFood                 CPIFood_MOM
	rename ConsumerPriceIndex36Cit    CPI36Cities
	rename PurchasingPriceIndexBeij   PPIBeijing
	rename PurchasingPriceIndexIndu   PPIIndustrial
	rename ImportsitcMFPetroleum      ImportPetroleum
	rename CentralBankBenchmarkInter  BenchmarkRate
	rename FXRatePBOCMonthEndRM       FXRate
	rename FixedAssetInvestmentytd    FAI_ytd
	rename FAIytdNewConstruction      FAIConstruction_ytd
	rename FAIytdEquipmentPurchase    FAIEquipment_ytd
	rename PMINonMfgBusinessActiv     PMI_nonmfg
	rename IndexShanghaiStockExchan   ShanghaiSE
	rename IndexShenzhenStockExchan   ShenzhenSE
	rename IndexShanghaiShenzhen300   ShanghaiShenzhen300
	rename PERatioShanghaiSEAllS      PEShanghaiSE
	rename PERatioShenzhenSEAllS      PEShenzhenSE
	rename RealEstateClimateIndexR    RealEstateClimate

	*drop duplicate variables
	*this variable is already in our dataset as "Retail"
	drop RetailSalesofConsumerGoo
	*This variable is already in our dataset as PMI
	drop PurchasingManagersIndex
	*this variable is already in our dataset as "FloorSpace"
	drop FloorSpaceStartedytdCo
	*this variable is already in our dataset as "Property_RealEstate"
	drop RealEstateInvestmentytd

	*drop older obs of EmpIndustrial, since they have too many missing values
	replace EmpIndustrial = "" if date<ym(2005,12)

	*label additional year-to-date variables
	rename FloorSpace FloorSpace_ytd
	rename Property_RealEstate Property_RealEstate_ytd
	rename ElectricityConsumption ElectricityConsumption_ytd

	*label year-over-year variables
	rename PPIBeijing    PPIBeijing_YOY
	rename PPIIndustrial PPIIndustrial_YOY

	*generate date variables
	ds date, not
	foreach v in `r(varlist)'{
		destring `v', replace
	}
	tsset date
	format date %tm
	gen year  = year(dofm(date))
	gen month = month(dofm(date))





	***************************
	* transform ytd variables into levels
	**************************
	local YTDVars ElectricityConsumption_ytd FloorSpace_ytd Property_RealEstate_ytd FAI_ytd FAIConstruction_ytd FAIEquipment_ytd 
	foreach v in `YTDVars'{

		*if January and February ytd values are equal, then discard the Janaury ytd value.
		replace `v' = . if month==1 & (`v'==F1.`v')
		
		*transform into levels. January level equal january ytd. February level equal february ytd only if january ytd is missing. Otherwise feb level equal febytd-janytd.
		gen     `v'2 = `v'        if month==1
		replace `v'2 = `v'        if month==2 & L1.`v'==.
		replace `v'2 = `v'-L1.`v' if month==2 & L1.`v'!=.
		replace `v'2 = `v'-L1.`v' if month!=1 & month!=2
		drop `v'
		rename `v'2 `v'
		
		*So right now, whenever Janaury values are missing, february values actually hold the sum of january and february.
		*we pick jan and feb levels such that jan+feb equals the value it's supposed to equal. Also, we pick the values such that Jan and Feb growth rates are equal
		*So this is a problem of two equations and two unknowns. The solution is our choice of values for janaury and february below.
		gen `v'2 = .
		gen a = L1.`v'
		gen b = F1.`v'
		
		replace `v'2 = `v'
		replace `v'2 = (-a+sqrt(a^2+4*a*b))/2 if month==1 & `v'==.
		replace `v'2 = `v'-L1.`v'2            if month==2 & L1.`v'==.
		
		
		drop `v' 
		rename `v'2 `v'
		
		drop a b
		
	}

	*rename ytd variables, since they are no longer year-to-date
	rename *_ytd *






	*****************************
	* Transform all variables into levels (except for the YOY variables)
	*******************************
	*note that some variables are in some sort of growth rate (the CPI variables)
	local LevelVars        GasConsumption ConsumerIndex Retail IP ElectricityConsumption ElectricityProduction Rail Property_RealEstate CrudeSteel Trucks FloorSpace Lending RawMaterials GovtRevenue GovtExpenditure GovtSurp GovtRevenueCentral GovtExpenditureCentral GovtSurpCentral HoldingsofUSTreasury EmpIndustrial ConsumerConfidenceIndex ExportFOB ImportCIF TradeBalance ImportPetroleum MoneySupplyM2 ForeignReserve GoldReserve FAI FAIConstruction FAIEquipment ShanghaiSE ShenzhenSE ShanghaiShenzhen300 RealEstateClimate Commodity Oil IP_US
	local CPIVarsMOM       CPI_MOM CPICore_MOM CPIFood_MOM CPI36Cities
	if `GrowthType'!=12{
		local DropVars     PPIBeijing_YOY PPIIndustrial_YOY
	}

	foreach v in `LevelVars'{
		*do nothing
	}
	foreach v in `CPIVarsMOM'{
		*convert to a price index
		replace `v' = `v'/100
		
		*identify the first observation
		gen flag = 1 if `v'!=.
		sum date if flag==1
		gen firstobsdate = `r(min)'
		gen firstobsflag = 1 if date==firstobsdate
		
		gen     `v'2 = .
		replace `v'2 = 1 if F1.firstobsflag==1
		replace `v'2 = L1.`v'2*`v' if F1.firstobsflag!=1
		
		drop `v'
		rename `v'2 `v'
		
		drop flag firstobsdate firstobsflag	
	}
	*we can't convert these variables into levels or MOM growth rates, so we drop them
	foreach v in `DropVars'{
		drop `v'
	}
	rename *_MOM *




	***************************
	* Adjust for Chinese New Year
	***************************
	if `ChineseNewYearAdj'==1{
		*don't adjust the following variables: GovtSurp  GovtSurpCentral TradeBalance
		ds date year month GovtSurp GovtSurpCentral TradeBalance, not
		foreach v in `r(varlist)'{

			*first, add January's value to February
			replace `v' = L1.`v'+`v' if month==2
			
			*next, set January's value to 0
			replace `v' = . if month==1
			
			*So right now, whenever Janaury values are missing, february values actually hold the sum of january and february.
			*we pick new jan and feb levels such that jan+feb equals the value it's supposed to equal. Also, we pick the values such that Jan and Feb growth rates are equal
			*So this is a problem of two equations and two unknowns. The solution is our choice of values for janaury and february below.
			gen `v'2 = .
			gen a = L1.`v'
			gen b = F1.`v'
			
			replace `v'2 = `v'
			replace `v'2 = (-a+sqrt(a^2+4*a*b))/2 if month==1 & `v'==.
			replace `v'2 = `v'-L1.`v'2            if month==2 & L1.`v'==.
			
			drop `v' 
			rename `v'2 `v'
			
			drop a b
			
		}
	}



	if `SeasonallyAdjust'==1{

		********************
		* Fill in data holes
		********************
		*We fill in the data holes with the levels that would give us constant growth rates. We observe data holes of size 1,2,and 3.
		local HoleySeries Retail IP GovtSurp GovtRevenueCentral GovtExpenditureCentral GovtSurpCentral EmpIndustrial RealEstateClimate
		foreach v in `HoleySeries'{

			gen `v'2 = .

			*fill in holes of size 1
			replace `v'2 = sqrt(L1.`v'*F1.`v') if `v'==.
			
			*fill in holes of size 2
			replace `v'2 = (L1.`v')^(2/3) * (F2.`v')^(1/3)  if `v'==. & F1.`v'==. & `v'2==.
			replace `v'2 = (L2.`v')^(1/3) * (F1.`v')^(2/3)  if `v'==. & L1.`v'==. & `v'2==.
			
			*fill in holes of size 3
			replace `v'2 = (L1.`v')^(3/4) * (F3.`v')^(1/4)  if `v'==. &                         F1.`v'==. & F2.`v'==.  & `v'2==.
			replace `v'2 = (L2.`v')^(1/2) * (F2.`v')^(1/2)  if `v'==. &             L1.`v'==. & F1.`v'==.              & `v'2==.
			replace `v'2 = (L3.`v')^(1/4) * (F1.`v')^(3/4)  if `v'==. & L2.`v'==. & L1.`v'==.                          & `v'2==.

			*fill in data holes in `v' using values in `v'2
			replace `v' = `v'2 if `v'==.
			
			drop `v'2
		}



		****************************************
		* Deseasonalize all the variables
		****************************************
		*produce deseasonalized variables. Don't deseasonalize GovtExpenditure yet because it is a special case. Also, don't deseasonalize the interest rate and reserve requirement variables, because we don't think they are seasonal.
		ds date year month GovtExpenditure RequiredReserveRatio NominalRate6mo NominalRate1yr BenchmarkRate Overnight BenchmarkRate20d, not
		foreach v in `r(varlist)'{
			disp "We're now on the following variable: `v'"
			
			preserve
				keep year month `v'
				
				*drop missing obs, since the x12 program can't handle that
				drop if `v'==.
				
				if "`v'"=="GovtSurp" | "`v'"=="GovtSurpCentral" | "`v'"=="TradeBalance"{
					x12 year month `v' 12, x11_options(mode=add)
				}
				else{
					x12 year month `v' 12
				}
				
				drop `v'

				save "../clean data/temp_files/`v'.dta", replace
			restore
		}
		
		*Note that GovtExpenditure is a special case. We will deseasonalize 1995-2009 and 2010-2013 separately.
		local v GovtExpenditure
		foreach range in 1995_2009 2010_2013{
			local begy = substr("`range'",1,4)
			local endy = substr("`range'",-4,4)
			
			preserve
				
				keep if year>=`begy' & year<=`endy'
				keep year month `v'
				drop if `v'==.
				
				x12 year month `v' 12
				
				save "../clean data/temp_files/`v'`range'.dta", replace
				
			restore
		}
		preserve
			local FirstFile = 1
			foreach range in 1995_2009 2010_2013{
				
				if `FirstFile'==1{
					use "../clean data/temp_files/`v'`range'.dta", clear
				}
				else{
					append using "../clean data/temp_files/`v'`range'.dta"
				}
			
				local FirstFile = 0
			}
			
			drop `v'
			save "../clean data/temp_files/`v'.dta", replace
		restore
		
		
		*Create temporary files for the variables we didn't seasonally adjust
		foreach v in GovtExpenditure RequiredReserveRatio NominalRate6mo NominalRate1yr BenchmarkRate Overnight BenchmarkRate20d{
			preserve
				keep year month `v'
				drop if `v'==.
				save "../clean data/temp_files/`v'.dta", replace
			restore
		}
		
		*merge deseasonalized variables back together
		ds date year month, not
		local AllVars `r(varlist)'
		keep date year month
		foreach v in `AllVars'{
			disp "We're now on variable: `v'"
			merge 1:1 year month using "../clean data/temp_files/`v'.dta"
			assert _merge!=2
			drop _merge
		}

		tsset date

		*forget the NSA series. keep only the SA series
		rename *_sa *
		
	}






	************************
	* transform seasonally adjusted series into month-over-month growth rates (or year-over-year growth rates if that's what was specified in the parameters at the top of the program)
	************************
	*Note that for interest rates and some other variables, we will leave them in levels
	local LevelVars        GasConsumption ConsumerIndex Retail IP ElectricityConsumption ElectricityProduction Rail Property_RealEstate CrudeSteel Trucks FloorSpace Lending RawMaterials GovtRevenue GovtExpenditure GovtSurp GovtRevenueCentral GovtExpenditureCentral GovtSurpCentral HoldingsofUSTreasury EmpIndustrial ConsumerConfidenceIndex ExportFOB ImportCIF TradeBalance ImportPetroleum MoneySupplyM2 ForeignReserve GoldReserve FAI FAIConstruction FAIEquipment ShanghaiSE ShenzhenSE ShanghaiShenzhen300 RealEstateClimate CPI CPICore CPIFood CPI36Cities Commodity Oil IP_US
	local KeepAsIs         RequiredReserveRatio NominalRate6mo NominalRate1yr BenchmarkRate Overnight BenchmarkRate20d FXRate PMI PMI_nonmfg PMI_Export PEShanghaiSE PEShenzhenSE 
	if `GrowthType'==12{
		local KeepAsIs `KeepAsIs' PPIBeijing_YOY PPIIndustrial_YOY
	}

	foreach v in `LevelVars'{
		disp "This variable is: `v'"
		gen     g`v' = .
		*note that we use log-changes rather than raw percent changes
		replace g`v' = 100*(log(`v')-log(L`GrowthType'.`v'))
		
		*replace the variable with its growth rate
		drop `v'
		rename g`v' `v'
	}

	foreach v in `KeepAsIs'{
		*do nothing to these variables. Keep them in levels, or whatever their units were in CEIC
	}





	******************************
	* Adjust for redefinition of M2 in 2011m10
	******************************
	local v MoneySupplyM2
	*compute the average growth rate of M2 in 2011m7-2011m12 excluding 2011m10. Replace the weird value for 2011m10 with this averaged value.
	sum `v' if date>=ym(2011,7) & date<=ym(2012,1) & date!=ym(2011,10)
	replace `v' = `r(mean)' if year==2011 & month==10





	***********************
	* Apply Biweight Filter
	***********************
	*keep only the dates we want. We drop the extra data now because the biweight filter is sensitive to which end points you use.
	keep if date>=ym(`StartY',`StartM')
	keep if date<=ym(`EndY',`EndM')

	*don't apply the filter to the interest rate variables
	ds date year month NominalRate6mo NominalRate1yr BenchmarkRate Overnight BenchmarkRate20d, not

	if `UseBW'==1{
		disp "applying biweight filter"
		foreach v in `r(varlist)'{
			
			disp "we're now on variable: `v'"
			preserve
				*export a temporary dataset with just the variable and the biweight parameter
				keep date `v'
				gen bw_parm = `BWParm'
				
				drop if `v'==.
				order date `v' bw_parm
				outsheet `v' bw_parm using "../clean data/temp_files/bw_input.csv", comma replace nonames
				
				*wait until the bw_input file is done being written
				capture confirm file "../clean data/temp_files/bw_input.csv"
				while _rc {
					sleep 1000
					capture confirm file "../clean data/temp_files/bw_input.csv"
				}

				*run matlab program to generate "bw_output.csv", which will contain the filtered variable
				local dir `c(pwd)'
				shell "matlab" -r "run('`dir'\apply_biweight_filter')"
				
				*tell Stata to wait until the matlab script is done and the bw_output.csv file is created. Stata will check every 1000 milliseconds that the file is created
				capture confirm file "../clean data/temp_files/bw_output.csv"
				while _rc {
					sleep 1000
					capture confirm file "../clean data/temp_files/bw_output.csv"
				}

				*save a temporary copy of the current data
				save "../clean data/temp_files/temp.dta", replace
				
				*read in trend series and save it
				insheet using "../clean data/temp_files/bw_output.csv", comma clear nonames
				merge 1:1 _n using "../clean data/temp_files/temp.dta"
				
				*delete the matlab input and output files
				erase "../clean data/temp_files/bw_input.csv"
				erase "../clean data/temp_files/bw_output.csv"
				sleep 2000
				
				*compute the filtered series as the original minus the trend
				gen v2 = `v'-v1
						
				*keep only date and the filtered series
				keep date v2
				rename v2 `v'
				save "../clean data/temp_files/temp.dta", replace
				
			restore
			
			drop `v'
			merge 1:1 date using "../clean data/temp_files/temp.dta"
			drop _merge
		
		}
		
	}






	*********************************************
	*format data and export as excel document
	*********************************************
	local GovtFinance      GovtRevenue GovtExpenditure GovtSurp GovtRevenueCentral GovtExpenditureCentral GovtSurpCentral HoldingsofUSTreasury 
	local LaborVars        EmpIndustrial 
	local HouseholdSurvey  ConsumerConfidenceIndex

	local PriceVars        CPI CPICore CPIFood CPI36Cities 
	if `GrowthType'==12{
		local PriceVars `PriceVars' PPIBeijing_YOY PPIIndustrial_YOY
	}

	local TradeVars        ExportFOB ImportCIF TradeBalance ImportPetroleum 
	local MoneyVars        MoneySupplyM2 ForeignReserve GoldReserve RequiredReserveRatio NominalRate6mo NominalRate1yr BenchmarkRate Overnight BenchmarkRate20d FXRate 
	local InvestmentVars   FAI FAIConstruction FAIEquipment PMI_nonmfg
	local FinancialVars    ShanghaiSE ShenzhenSE ShanghaiShenzhen300 PEShanghaiSE PEShenzhenSE 
	local RealEstateVars   RealEstateClimate
	local ForeignShockVars Commodity Oil IP_US

	order date year month `BroadVars' `GovtFinance' `LaborVars' `HouseholdSurvey' `PriceVars' `TradeVars' `MoneyVars' `InvestmentVars' `FinancialVars' `RealEstateVars'
	export excel using "../clean data/transformed data (`MOMorYOY' `CNYadj'`BW'`BWParm' `StartY'`StartM'_`EndY'`EndM').xlsx", firstrow(var) sheetreplace sheet("data") datestring(%tm)
	save "../clean data/transformed data (`MOMorYOY' `CNYadj'`BW'`BWParm' `StartY'`StartM'_`EndY'`EndM').dta", replace

	
end


*Make three different versions of the cleaned data (we'll use the extra versions for robustness checks)
MakeData 1 1 36 2000 1 2013 10
MakeData 1 0 "" 2000 1 2013 10
MakeData 0 1 36 2000 1 2013 10
